<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';
require_once '../includes/functions.php';

// Helper function to build redirect URL with current filters
function buildRedirectUrl($base_url = "products.php") {
    $params = [];
    
    // Check for filters in POST data first (from forms), then fall back to GET
    $category_id = $_POST['filter_category_id'] ?? $_GET['category_id'] ?? '';
    $search = $_POST['filter_search'] ?? $_GET['search'] ?? '';
    $sort = $_POST['filter_sort'] ?? $_GET['sort'] ?? '';
    $order = $_POST['filter_order'] ?? $_GET['order'] ?? '';
    
    if (!empty($category_id)) {
        $params['category_id'] = $category_id;
    }
    if (!empty($search)) {
        $params['search'] = $search;
    }
    if (!empty($sort)) {
        $params['sort'] = $sort;
    }
    if (!empty($order)) {
        $params['order'] = $order;
    }
    
    if (!empty($params)) {
        return $base_url . '?' . http_build_query($params);
    }
    
    return $base_url;
}

// Set active page for navigation
$activePage = 'products';

// Check if user is logged in
requireLogin();

// Check if user has any product-related permissions
$productPermissions = [
    'view_products', 'manage_products', 'add_product', 'edit_product', 'delete_product',
    'view_inventory', 'manage_inventory', 'add_stock', 'edit_stock', 'delete_stock'
];

if (!hasAnyPermission($productPermissions)) {
    $_SESSION['error'] = "You don't have permission to access product management.";
    header("Location: ../index.php");
    exit;
}

$success = '';
$error = '';
$search = $_GET['search'] ?? '';
$category_id = $_GET['category_id'] ?? '';
$sort = $_GET['sort'] ?? 'name';
$order = $_GET['order'] ?? 'asc';
$adjust_success = false;

// Get all suppliers
$stmt = $pdo->query("SELECT id, name FROM suppliers ORDER BY name");
$suppliers = $stmt->fetchAll();

// Get all categories with their require_serials setting
$stmt = $pdo->query("SELECT id, name, description, require_serials FROM categories ORDER BY name");
$categories = $stmt->fetchAll();

// Create a lookup array for categories
$categories_by_id = [];
foreach ($categories as $category) {
    $categories_by_id[$category['id']] = $category;
}

// Build the products query with filtering
$query = "
    SELECT p.*, c.name as category_name, c.require_serials,
           (SELECT COUNT(*) FROM sale_items si WHERE si.product_id = p.id) as times_sold,
           (SELECT GROUP_CONCAT(psn.serial_number SEPARATOR ', ') 
            FROM product_serial_numbers psn 
            WHERE psn.product_id = p.id AND psn.status = 'available') as serial_numbers
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
    WHERE 1=1
";

$params = [];

// Apply filters
if ($search) {
    $query .= " AND (p.name LIKE ? OR p.description LIKE ? OR 
                     EXISTS (SELECT 1 FROM product_serial_numbers psn WHERE psn.product_id = p.id AND psn.serial_number LIKE ?))";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($category_id) {
    $query .= " AND p.category_id = ?";
    $params[] = $category_id;
}

// Add sorting
$valid_sorts = ['name', 'price', 'cost_price', 'quantity', 'times_sold'];
$sort_column = in_array($sort, $valid_sorts) ? $sort : 'name';
if ($sort_column === 'times_sold') {
    $query .= " ORDER BY times_sold " . ($order === 'desc' ? 'DESC' : 'ASC') . ", p.name ASC";
} else {
    $query .= " ORDER BY p.$sort_column " . ($order === 'desc' ? 'DESC' : 'ASC');
}

// Get filtered products
try {
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $products = $stmt->fetchAll();
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Calculate statistics
$total_products = count($products);
$total_value = 0;
$low_stock = 0;
$out_of_stock = 0;

foreach ($products as $product) {
    $total_value += $product['quantity'] * $product['price'];
    
    if ($product['quantity'] == 0) {
        $out_of_stock++;
    } elseif ($product['quantity'] <= $product['alert_quantity']) {
        $low_stock++;
    }
}

// For dropdown (includes require_serials)
try {
    $stmt = $pdo->query("
        SELECT p.*, c.require_serials 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        ORDER BY p.name
    ");
    $products_for_dropdown = $stmt->fetchAll();
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

// Handle product actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_product'])) {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO products (name, description, category_id, price, cost_price, quantity, alert_quantity)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['name'],
                $_POST['description'],
                $_POST['category_id'],
                $_POST['price'],
                $_POST['cost_price'],
                $_POST['quantity'],
                $_POST['alert_quantity']
            ]);
            
            $product_id = $pdo->lastInsertId();
            
            // Handle serial numbers if quantity > 0
            if ($_POST['quantity'] > 0 && !empty($_POST['serial_numbers'])) {
                $serialNumbers = array_map('trim', explode("\n", str_replace(',', "\n", $_POST['serial_numbers'])));
                $stmt = $pdo->prepare("
                    INSERT INTO product_serial_numbers (product_id, serial_number, status, source_type, source_id)
                    VALUES (?, ?, 'available', 'manual_adjustment', ?)
                ");
                foreach ($serialNumbers as $serial) {
                    if (!empty($serial)) {
                        $stmt->execute([$product_id, $serial, $_SESSION['user_id']]);
                    }
                }
            }
            
            $_SESSION['success'] = "Product added successfully.";
            header("Location: " . buildRedirectUrl());
            exit();
        } catch (PDOException $e) {
            $_SESSION['error'] = "Error adding product: " . $e->getMessage();
        }
    } elseif (isset($_POST['edit_product'])) {
        try {
            $stmt = $pdo->prepare("
                UPDATE products 
                SET name = ?, description = ?, category_id = ?, price = ?, cost_price = ?, 
                    quantity = ?, alert_quantity = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $_POST['name'],
                $_POST['description'],
                $_POST['category_id'],
                $_POST['price'],
                $_POST['cost_price'],
                $_POST['quantity'],
                $_POST['alert_quantity'],
                $_POST['id']
            ]);

            // Handle serial numbers if provided
            if (!empty($_POST['serial_numbers'])) {
                $serialNumbers = array_map('trim', explode("\n", str_replace(',', "\n", $_POST['serial_numbers'])));
                // Delete existing serial numbers
                $stmt = $pdo->prepare("DELETE FROM product_serial_numbers WHERE product_id = ?");
                $stmt->execute([$_POST['id']]);
                
                // Insert new serial numbers
                $stmt = $pdo->prepare("
                    INSERT INTO product_serial_numbers (product_id, serial_number, status, source_type, source_id)
                    VALUES (?, ?, 'available', 'manual_adjustment', ?)
                ");
                foreach ($serialNumbers as $serial) {
                    if (!empty($serial)) {
                        $stmt->execute([$_POST['id'], $serial, $_SESSION['user_id']]);
                    }
                }
            }
            
            $_SESSION['success'] = "Product updated successfully.";
            header("Location: " . buildRedirectUrl());
            exit();
        } catch (PDOException $e) {
            $_SESSION['error'] = "Error updating product: " . $e->getMessage();
        }
    } elseif (isset($_POST['delete_product'])) {
        try {
            // Check if product has any sales
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM sale_items WHERE product_id = ?");
            $stmt->execute([intval($_POST['id'])]);
            $hasSales = $stmt->fetchColumn() > 0;
            if ($hasSales) {
                $error = "Cannot delete product with existing sales records.";
            } else {
                $stmt = $pdo->prepare("DELETE FROM products WHERE id = ?");
                $stmt->execute([intval($_POST['id'])]);
                $_SESSION['success'] = "Product deleted successfully!";
                header("Location: " . buildRedirectUrl());
                exit();
            }
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), 'a foreign key constraint fails') !== false) {
                $error = "Cannot delete this product because it is referenced in other records (e.g., supplier receipts or sales). Please remove all related records first.";
            } else {
                $error = "Error deleting product: " . $e->getMessage();
            }
        }

    } elseif (isset($_POST['add_category'])) {
        try {
            // Check if category name already exists
            $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM categories WHERE name = ?");
            $checkStmt->execute([trim($_POST['category_name'])]);
            
            if ($checkStmt->fetchColumn() > 0) {
                $error = "A category with this name already exists.";
            } else {
                $stmt = $pdo->prepare("INSERT INTO categories (name, description, require_serials) VALUES (?, ?, ?)");
                $stmt->execute([
                    trim($_POST['category_name']),
                    !empty($_POST['category_description']) ? trim($_POST['category_description']) : null,
                    isset($_POST['require_serials']) ? intval($_POST['require_serials']) : 0
                ]);
                $success = "Category added successfully!";
                header('Location: ' . buildRedirectUrl());
                exit();
            }
        } catch (PDOException $e) {
            $error = "Error adding category: " . $e->getMessage();
        }
    } elseif (isset($_POST['edit_category'])) {
        try {
            $stmt = $pdo->prepare("UPDATE categories SET name = ?, description = ?, require_serials = ? WHERE id = ?");
            $stmt->execute([
                trim($_POST['category_name']),
                !empty($_POST['category_description']) ? trim($_POST['category_description']) : null,
                isset($_POST['require_serials']) ? intval($_POST['require_serials']) : 0,
                intval($_POST['category_id'])
            ]);
            $success = "Category updated successfully!";
            header('Location: ' . buildRedirectUrl());
            exit();
        } catch (PDOException $e) {
            $error = "Error updating category: " . $e->getMessage();
        }
    } elseif (isset($_POST['delete_category'])) {
        try {
            // Check if category has any products
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM products WHERE category_id = ?");
            $stmt->execute([intval($_POST['category_id'])]);
            $hasProducts = $stmt->fetchColumn() > 0;

            if ($hasProducts) {
                $error = "Cannot delete category with existing products.";
            } else {
                $stmt = $pdo->prepare("DELETE FROM categories WHERE id = ?");
                $stmt->execute([intval($_POST['category_id'])]);
                $success = "Category deleted successfully!";
            }
        } catch (PDOException $e) {
            $error = "Error deleting category: " . $e->getMessage();
        }
    } elseif (isset($_POST['adjust_stock'])) {
        echo '<div style="position:fixed;top:0;left:0;z-index:99999;background:#ffc;padding:10px;border:2px solid #f00;">DEBUG: Entered adjust_stock handler</div>';
        error_log('DEBUG: Entered adjust_stock handler');
        $product_id = intval($_POST['product_id']);
        $quantity = intval($_POST['quantity']);
        $supplier_id = intval($_POST['supplier_id'] ?? 0);
        $unit_price = floatval($_POST['unit_price'] ?? 0);
        $serial_numbers = isset($_POST['serial_numbers']) ? array_map('trim', explode("\n", str_replace(',', "\n", $_POST['serial_numbers']))) : [];
        $batch_number = 'manual';
        
        // Fetch product category_id for require_serials check
        $stmt = $pdo->prepare("SELECT category_id FROM products WHERE id = ?");
        $stmt->execute([$product_id]);
        $product_row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($product_row) {
            $category_id = $product_row['category_id'];
        } else {
            $error = "Product not found.";
            echo '<div style="position:fixed;top:30px;left:0;z-index:99999;background:#fcc;padding:10px;border:2px solid #f00;">DEBUG: Product not found</div>';
            error_log('DEBUG: Product not found');
        }
        // Fetch require_serials for the category
        $require_serials = 0;
        if (!empty($category_id)) {
            $stmt = $pdo->prepare("SELECT require_serials FROM categories WHERE id = ?");
            $stmt->execute([$category_id]);
            $require_serials = (int)($stmt->fetchColumn());
        }
        if ($product_id && $quantity && $supplier_id && empty($error)) {
            try {
                $pdo->beginTransaction();
                // 1. Insert into supplier_receipts
                $serials_json = null;
                if ($require_serials && !empty($serial_numbers)) {
                    $serials_json = json_encode($serial_numbers);
                }
                $stmt = $pdo->prepare("
                    INSERT INTO supplier_receipts (
                        supplier_id, product_id, quantity, unit_price, batch_number, created_by, created_at, serial_numbers
                    ) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)
                ");
                $stmt->execute([
                    $supplier_id,
                    $product_id,
                    $quantity,
                    $unit_price,
                    $batch_number,
                    $_SESSION['user_id'],
                    $serials_json
                ]);
                error_log('DEBUG: Inserted into supplier_receipts');
                // 2. Update product quantity
                $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
                $stmt->execute([$quantity, $product_id]);
                error_log('DEBUG: Updated products quantity');
                // 3. Add serial numbers if required
                if ($require_serials && $quantity > 0 && !empty($serial_numbers)) {
                    if (count($serial_numbers) !== $quantity) {
                        throw new Exception("Number of serial numbers must match the adjustment quantity.");
                    }
                    $stmt = $pdo->prepare("
                        INSERT INTO product_serial_numbers (product_id, serial_number, status, source_type, source_id)
                        VALUES (?, ?, 'available', 'manual_adjustment', ?)
                    ");
                    foreach ($serial_numbers as $serial) {
                        if (!empty($serial)) {
                            $stmt->execute([$product_id, $serial, $_SESSION['user_id']]);
                        }
                    }
                    error_log('DEBUG: Inserted serial numbers');
                }
                // 4. Log stock adjustment
                $stmt = $pdo->prepare("
                    INSERT INTO stock_adjustments (product_id, quantity, reason, user_id, supplier_id)
                    VALUES (?, ?, ?, ?, ?)
                ");
                $stmt->execute([$product_id, $quantity, 'Manual stock adjustment', $_SESSION['user_id'], $supplier_id]);
                error_log('DEBUG: Inserted into stock_adjustments');
                $pdo->commit();
                $_SESSION['adjust_success'] = true;
                header('Location: ' . buildRedirectUrl());
                exit();
            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error adjusting stock: " . $e->getMessage();
                echo '<div style="position:fixed;top:60px;left:0;z-index:99999;background:#fcc;padding:10px;border:2px solid #f00;">DEBUG: Exception - ' . htmlspecialchars($e->getMessage()) . '</div>';
                error_log('DEBUG: Exception - ' . $e->getMessage());
            }
        } else {
            $error = "Please fill in all required fields.";
            echo '<div style="position:fixed;top:90px;left:0;z-index:99999;background:#fcc;padding:10px;border:2px solid #f00;">DEBUG: Validation failed</div>';
            error_log('DEBUG: Validation failed');
        }
    } elseif (isset($_POST['add_product_name'])) {
        try {
            // Check if product name already exists (case-insensitive)
            $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM products WHERE LOWER(name) = LOWER(?)");
            $checkStmt->execute([trim($_POST['name'])]);
            
            if ($checkStmt->fetchColumn() > 0) {
                $_SESSION['error'] = "A product with this name already exists.";
            } else {
                $stmt = $pdo->prepare("INSERT INTO products (name, description, category_id, price, cost_price, quantity, alert_quantity) VALUES (?, ?, ?, 0.00, 0.00, 0, 0)");
                $stmt->execute([
                    trim($_POST['name']),
                    trim($_POST['description']),
                    intval($_POST['category_id'])
                ]);
                $_SESSION['success'] = "Product name added successfully!";
            }
            // Redirect to prevent form resubmission
            header('Location: ' . buildRedirectUrl());
            exit;
        } catch (PDOException $e) {
            $_SESSION['error'] = "Error adding product name: " . $e->getMessage();
            header('Location: ' . buildRedirectUrl());
            exit;
        }
    } elseif (isset($_POST['adjust_stock_multiple'])) {
        $supplier_id = intval($_POST['supplier_id'] ?? 0);
        $products_data = $_POST['products'] ?? [];
        
        // Generate the next batch number and add the "MM_" prefix
        $batch_number = 'MM_' . getNextBatchNumber($pdo);

        if (!$supplier_id || empty($products_data)) {
            $error = "Please select a supplier and add at least one product.";
        } else {
            try {
                $pdo->beginTransaction();
                $success_count = 0;
                $error_messages = [];
                
                foreach ($products_data as $index => $product_data) {
                    $product_id = intval($product_data['product_id'] ?? 0);
                    $quantity = intval($product_data['quantity'] ?? 0);
                    $unit_price = floatval($product_data['unit_price'] ?? 0);
                    $serial_numbers = isset($product_data['serial_numbers']) ? array_map('trim', explode("\n", str_replace(',', "\n", $product_data['serial_numbers']))) : [];
                    
                    if (!$product_id || $quantity == 0 || $unit_price <= 0) {
                        $error_messages[] = "Product #" . ($index + 1) . ": Invalid data provided";
                        continue;
                    }
                    
                    // Fetch product category_id for require_serials check
                    $stmt = $pdo->prepare("SELECT category_id FROM products WHERE id = ?");
                    $stmt->execute([$product_id]);
                    $product_row = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$product_row) {
                        $error_messages[] = "Product #" . ($index + 1) . ": Product not found";
                        continue;
                    }
                    
                    $category_id = $product_row['category_id'];
                    
                    // Fetch require_serials for the category
                    $require_serials = 0;
                    if (!empty($category_id)) {
                        $stmt = $pdo->prepare("SELECT require_serials FROM categories WHERE id = ?");
                        $stmt->execute([$category_id]);
                        $require_serials = (int)($stmt->fetchColumn());
                    }
                    
                    // Validate serial numbers if required
                    if ($require_serials && $quantity > 0 && !empty($serial_numbers)) {
                        if (count($serial_numbers) !== $quantity) {
                            $error_messages[] = "Product #" . ($index + 1) . ": Number of serial numbers must match the adjustment quantity";
                            continue;
                        }
                    }
                    
                    // 1. Insert into supplier_receipts
                    $serials_json = null;
                    if ($require_serials && !empty($serial_numbers)) {
                        $serials_json = json_encode($serial_numbers);
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_receipts (
                            supplier_id, product_id, quantity, unit_price, batch_number, created_by, created_at, serial_numbers, status, added_to_stock
                        ) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, 'yes', 1)
                    ");
                    $stmt->execute([
                        $supplier_id,
                        $product_id,
                        $quantity,
                        $unit_price,
                        $batch_number, // Use the unique batch number
                        $_SESSION['user_id'],
                        $serials_json
                    ]);
                    
                    // 2. Update product quantity
                    $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
                    $stmt->execute([$quantity, $product_id]);
                    
                    // 3. Add serial numbers if required
                    if ($require_serials && $quantity > 0 && !empty($serial_numbers)) {
                        $stmt = $pdo->prepare("
                            INSERT INTO product_serial_numbers (product_id, serial_number, status, source_type, source_id)
                            VALUES (?, ?, 'available', 'manual_adjustment', ?)
                        ");
                        foreach ($serial_numbers as $serial) {
                            if (!empty($serial)) {
                                $stmt->execute([$product_id, $serial, $_SESSION['user_id']]);
                            }
                        }
                    }
                    
                    // 4. Log stock adjustment
                    $stmt = $pdo->prepare("
                        INSERT INTO stock_adjustments (product_id, quantity, reason, user_id, supplier_id)
                        VALUES (?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$product_id, $quantity, 'Manual stock adjustment (multiple)', $_SESSION['user_id'], $supplier_id]);
                    
                    $success_count++;
                }
                
                if ($success_count > 0) {
                    $pdo->commit();
                    $_SESSION['success'] = "Successfully updated stock for {$success_count} product(s).";
                    if (!empty($error_messages)) {
                        $_SESSION['error'] = "Some products failed to update: " . implode(", ", $error_messages);
                    }
                } else {
                    $pdo->rollBack();
                    $error = "No products were updated. Errors: " . implode(", ", $error_messages);
                }
                
                header('Location: ' . buildRedirectUrl());
                exit();
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error adjusting stock: " . $e->getMessage();
            }
        }
    }
}

// Display success/error messages from session
if (isset($_SESSION['success'])) {
    $success = $_SESSION['success'];
    unset($_SESSION['success']);
}
if (isset($_SESSION['error'])) {
    $error = $_SESSION['error'];
    unset($_SESSION['error']);
}

// Fetch products with category names and available serial numbers
$stmt = $pdo->query("
    SELECT p.*, c.name as category_name,
           GROUP_CONCAT(psn.serial_number) as serial_numbers,
           (SELECT COUNT(*) FROM sale_items si WHERE si.product_id = p.id) as times_sold
          FROM products p
          LEFT JOIN categories c ON p.category_id = c.id
    LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id AND psn.status = 'available'
    GROUP BY p.id
    ORDER BY p.name ASC
");
$products = $stmt->fetchAll();

// Group serial numbers by product ID for display
$serials_by_product = [];
foreach ($products as $product) {
    $serials_by_product[$product['id']] = $product['serial_numbers'] ?? '';
}

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Calculate statistics
$total_products = count($products);
$total_value = 0;
$low_stock = 0;
$out_of_stock = 0;

foreach ($products as $product) {
    $total_value += $product['price'] * $product['quantity'];
    if ($product['quantity'] <= $product['alert_quantity']) {
        $low_stock++;
    }
    if ($product['quantity'] == 0) {
        $out_of_stock++;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - EDDUH COMPUTERS</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { 
            background: #f8fafc; 
            font-family: 'Inter', sans-serif; 
            position: relative
        }
        .card { 
            border-radius: 16px; 
            box-shadow: 0 4px 20px rgba(0,0,0,0.05); 
            border: none; 
            transition: transform 0.2s;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .card-header { 
            background: none; 
            border-bottom: none; 
            padding: 1.5rem 1.5rem 0.5rem;
        }
        .card-body {
            padding: 1.5rem;
        }
        .stat-card {
            background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
            border-left: 4px solid;
        }
        .stat-card.primary { border-left-color: #3b82f6; }
        .stat-card.success { border-left-color: #10b981; }
        .stat-card.info { border-left-color: #06b6d4; }
        .stat-card.warning { border-left-color: #f59e0b; }
        .stat-value {
            font-size: 1.8rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0.5rem 0;
        }
        .stat-label {
            color: #64748b;
            font-size: 0.9rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }
        .table {
            background: #fff;
            border-radius: 12px;
            overflow: hidden;
        }
        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: #475569;
            border-bottom: 2px solid #e2e8f0;
        }
        .table td {
            vertical-align: middle;
            color: #334155;
        }
        .btn {
            border-radius: 8px;
            font-weight: 500;
            padding: 0.5rem 1rem;
        }
        .btn-primary {
            background: #3b82f6;
            border: none;
        }
        .btn-primary:hover {
            background: #2563eb;
        }
        .filter-card {
            background: #fff;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
        }
        #productsContainer {
            scrollbar-width: thin;
            scrollbar-color: #cbd5e0 #f7fafc;
        }
        #productsContainer::-webkit-scrollbar {
            width: 8px;
        }
        #productsContainer::-webkit-scrollbar-track {
            background: #f7fafc;
            border-radius: 4px;
        }
        #productsContainer::-webkit-scrollbar-thumb {
            background: #cbd5e0;
            border-radius: 4px;
        }
        #productsContainer::-webkit-scrollbar-thumb:hover {
            background: #a0aec0;
        }


        .product-entry {
            border: 1px solid #e2e8f0 !important;
            border-radius: 8px !important;
            padding: 1rem !important;
            margin-bottom: 1rem !important;
            background: #fff;
            transition: box-shadow 0.2s ease;
        }
        .product-entry:hover {
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .filter-card .form-control {
            border-radius: 6px;
            border: 1px solid #e2e8f0;
        }
        .filter-card .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .badge {
            padding: 0.5em 1em;
            border-radius: 6px;
            font-weight: 500;
        }
        .badge-stock {
            background: #10b981;
            color: white;
        }
        .badge-low {
            background: #f59e0b;
            color: white;
        }
        .badge-out {
            background: #ef4444;
            color: white;
        }
        .progress {
            height: 8px;
            border-radius: 4px;
            background: #e2e8f0;
        }
        .progress-bar {
            border-radius: 4px;
        }
        .modal-content {
            border-radius: 16px;
            border: none;
        }
        .modal-header {
            border-bottom: none;
            padding: 1.5rem;
        }
        .modal-body {
            padding: 1.5rem;
        }
        .modal-footer {
            border-top: none;
            padding: 1.5rem;
        }
        .hover-bg-light:hover {
            background-color: #f8f9fa;
            cursor: pointer;
        }
        .cursor-pointer {
            cursor: pointer;
        }
        .supplier-search-item {
            cursor: pointer;
            padding: 0.75rem 1rem;
            border-bottom: 1px solid #e9ecef;
        }
        .supplier-search-item:hover {
            background-color: #f8f9fa;
        }
        .supplier-search-item.active {
            background-color: #e9ecef;
        }
        .supplier-search-item .supplier-name {
            font-weight: 600;
            color: #212529;
        }
        .supplier-search-item .supplier-details {
            font-size: 0.875rem;
            color: #6c757d;
        }
        #supplierSearchResults {
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            background-color: white;
        }
        .modal-lg {
            max-width: 900px;
        }
        .card {
            border: 1px solid rgba(0,0,0,.125);
            box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,.075);
        }
        .card-header {
            background-color: #f8f9fa;
            border-bottom: 1px solid rgba(0,0,0,.125);
        }
        .form-label {
            font-weight: 500;
            margin-bottom: 0.5rem;
        }
        .input-group-text {
            background-color: #f8f9fa;
        }
        .btn {
            padding: 0.5rem 1rem;
            font-weight: 500;
        }
        .btn i {
            margin-right: 0.5rem;
        }
        .modal-header {
            border-bottom: none;
            padding: 1rem 1.5rem;
        }
        .modal-footer {
            border-top: none;
            padding: 1rem 1.5rem;
        }
        .modal-body {
            padding: 1.5rem;
        }
        .invalid-feedback {
            font-size: 0.875rem;
        }
        /* Custom scrollbar for textarea */
        textarea {
            scrollbar-width: thin;
            scrollbar-color: #dee2e6 #fff;
        }
        textarea::-webkit-scrollbar {
            width: 8px;
        }
        textarea::-webkit-scrollbar-track {
            background: #fff;
        }
        textarea::-webkit-scrollbar-thumb {
            background-color: #dee2e6;
            border-radius: 4px;
        }
        .modal-dialog {
            max-width: 500px;
        }
        .modal-body {
            padding: 1rem;
        }
        .form-label {
            font-weight: 500;
            margin-bottom: 0.25rem;
            font-size: 0.9rem;
        }
        .input-group-text {
            background-color: #f8f9fa;
        }
        .btn {
            padding: 0.375rem 0.75rem;
            font-size: 0.9rem;
        }
        .btn i {
            margin-right: 0.25rem;
        }
        .modal-header {
            padding: 0.75rem 1rem;
        }
        .modal-footer {
            padding: 0.75rem 1rem;
        }
        .invalid-feedback {
            font-size: 0.8rem;
        }
        textarea {
            font-size: 0.9rem;
        }
        .form-control, .form-select {
            font-size: 0.9rem;
            padding: 0.375rem 0.75rem;
        }
        .serial-numbers-container {
            margin-top: 0.5rem;
        }
        .serial-numbers-container textarea {
            font-family: monospace;
            font-size: 0.9rem;
        }
        .form-text {
            font-size: 0.8rem;
            margin-top: 0.25rem;
        }
        .alert {
            padding: 0.75rem;
            margin-bottom: 0;
            font-size: 0.9rem;
        }
        .alert i {
            margin-right: 0.5rem;
        }
        .modal-body {
            padding: 1.5rem;
        }
        .modal-footer {
            border-top: none;
            padding: 1.5rem;
        }
        .product-entry {
            background: #f8fafc;
            border: 1px solid #e2e8f0 !important;
            transition: all 0.2s ease;
        }
        /* Categories table scrollable after 5 rows */
        .categories-table-container {
            max-height: 320px; /* Approximately 5 rows + header */
            overflow-y: auto;
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.08);
        }
        .categories-table-container .table {
            margin-bottom: 0;
        }
        .categories-table-container thead th {
            position: sticky;
            top: 0;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            z-index: 10;
            border-bottom: 2px solid #dee2e6;
            font-weight: 600;
            color: #495057;
            padding: 12px 8px;
        }
        
        /* Beautiful Category Modal Styles */
        #categoryModal .modal-content {
            border: none;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.15);
            overflow: hidden;
        }
        
        #categoryModal .modal-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 20px 30px;
            position: relative;
        }
        
        #categoryModal .modal-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" patternUnits="userSpaceOnUse" width="100" height="100"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="20" cy="80" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="30" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            pointer-events: none;
        }
        
        #categoryModal .modal-title {
            font-weight: 700;
            font-size: 1.4rem;
            position: relative;
            z-index: 1;
        }
        
        #categoryModal .btn-close {
            position: relative;
            z-index: 1;
            background: rgba(255,255,255,0.2);
            border-radius: 50%;
            width: 35px;
            height: 35px;
            opacity: 1;
        }
        
        #categoryModal .btn-close:hover {
            background: rgba(255,255,255,0.3);
            transform: scale(1.1);
        }
        
        #categoryModal .modal-body {
            padding: 30px;
            background: #fafbfc;
        }
        
        .category-form-section {
            background: white;
            border-radius: 12px;
            padding: 18px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
            border: 1px solid rgba(102,126,234,0.1);
            position: relative;
            overflow: hidden;
        }
        
        .category-form-section::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
        }
        
        .category-form-section h6 {
            color: #495057;
            font-weight: 700;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 1.1rem;
        }
        
        .category-form-section .form-label {
            font-weight: 600;
            color: #495057;
            margin-bottom: 5px;
            font-size: 0.9rem;
        }
        
        .category-form-section .form-control,
        .category-form-section .form-select {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 8px 12px;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            background-color: #fff;
        }
        
        .category-form-section .form-control:focus,
        .category-form-section .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 4px rgba(102,126,234,0.1);
            background-color: #fff;
        }
        
        .category-form-section .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            border-radius: 8px;
            padding: 8px 16px;
            font-weight: 600;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(102,126,234,0.3);
        }
        
        .category-form-section .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102,126,234,0.4);
        }
        
        .categories-list-section {
            background: white;
            border-radius: 12px;
            padding: 18px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
            border: 1px solid rgba(102,126,234,0.1);
            position: relative;
            overflow: hidden;
        }
        
        .categories-list-section::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #764ba2 0%, #667eea 100%);
        }
        
        .categories-list-section h6 {
            color: #495057;
            font-weight: 700;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 1.1rem;
        }
        
        .categories-table-container {
            max-height: 300px;
            overflow-y: auto;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            box-shadow: inset 0 1px 3px rgba(0,0,0,0.04);
        }
        
        .categories-table-container .table {
            margin-bottom: 0;
            border-collapse: separate;
            border-spacing: 0;
        }
        
        .categories-table-container thead th {
            position: sticky;
            top: 0;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            z-index: 10;
            border: none;
            border-bottom: 2px solid #dee2e6;
            font-weight: 700;
            color: #495057;
            padding: 12px 10px;
            font-size: 0.85rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .categories-table-container tbody tr {
            transition: all 0.2s ease;
        }
        
        .categories-table-container tbody tr:hover {
            background-color: rgba(102,126,234,0.04);
            transform: translateX(2px);
        }
        
        .categories-table-container tbody td {
            padding: 10px 10px;
            border: none;
            border-bottom: 1px solid #f1f3f4;
            vertical-align: middle;
        }
        
        .categories-table-container .btn {
            border-radius: 6px;
            font-weight: 500;
            font-size: 0.8rem;
            padding: 4px 8px;
            transition: all 0.2s ease;
        }
        
        .categories-table-container .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            box-shadow: 0 2px 6px rgba(102,126,234,0.3);
        }
        
        .categories-table-container .btn-primary:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(102,126,234,0.4);
        }
        
        .categories-table-container .btn-danger {
            background: linear-gradient(135deg, #ff6b6b 0%, #ee5a52 100%);
            border: none;
            box-shadow: 0 2px 6px rgba(255,107,107,0.3);
        }
        
        .categories-table-container .btn-danger:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(255,107,107,0.4);
        }
        
        /* Custom scrollbar for categories table */
        .categories-table-container::-webkit-scrollbar {
            width: 8px;
        }
        
        .categories-table-container::-webkit-scrollbar-track {
            background: #f1f3f4;
            border-radius: 4px;
        }
        
        .categories-table-container::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 4px;
        }
        
        .categories-table-container::-webkit-scrollbar-thumb:hover {
            background: linear-gradient(135deg, #5a6fd8 0%, #6a4190 100%);
        }
        
        /* Empty state styling */
        .categories-empty-state {
            text-align: center;
            padding: 30px 15px;
            color: #6c757d;
        }
        
        .categories-empty-state i {
            font-size: 2.5rem;
            margin-bottom: 10px;
            opacity: 0.5;
        }
        
        /* Modal Animation Effects */
        #categoryModal.fade .modal-dialog {
            transform: scale(0.8) translateY(-50px);
            transition: all 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
        }
        
        #categoryModal.show .modal-dialog {
            transform: scale(1) translateY(0);
        }
        
        /* Badge styling improvements */
        .badge {
            font-size: 0.75rem;
            padding: 0.4em 0.8em;
            border-radius: 8px;
            font-weight: 500;
        }
        
        /* Button group improvements */
        .btn-group .btn {
            margin-right: 0;
        }
        
        .btn-group .btn:first-child {
            margin-right: 2px;
        }
        
        /* Form improvements */
        .category-form-section textarea {
            resize: vertical;
            min-height: 80px;
        }
        
        .category-form-section .form-control::placeholder,
        .category-form-section .form-select::placeholder {
            color: #adb5bd;
            font-style: italic;
        }
        
        /* Hover effects for table rows */
        .categories-table-container tbody tr:hover td {
            background-color: rgba(102,126,234,0.06);
        }
        
        /* Success animation for form submission */
        @keyframes successPulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.02); }
            100% { transform: scale(1); }
        }
        
        .category-form-section.success {
            animation: successPulse 0.6s ease-in-out;
        }
        .product-entry:hover {
            background: #f1f5f9;
            border-color: #cbd5e1 !important;
        }
        .product-entry h6 {
            color: #475569;
            font-weight: 600;
        }
        .remove-product {
            border-radius: 50%;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 0;
        }
        .remove-product:hover {
            background-color: #ef4444;
            border-color: #ef4444;
            color: white;
        }
        .category-display, .current-stock-display {
            background-color: #f8fafc;
            color: #64748b;
        }
        .serial-numbers-section {
            background: #fef3c7;
            border: 1px solid #f59e0b;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
        }
        .serial-numbers-section label {
            color: #92400e;
            font-weight: 600;
        }
        #addProductBtn {
            border-radius: 8px;
            font-weight: 500;
        }
        #addProductBtn:hover {
            background-color: #3b82f6;
            border-color: #3b82f6;
            color: white;
        }
        
        /* Bulk Update Styles */
        .bulk-update-dropdown .dropdown-menu {
            min-width: 250px;
        }
        
        .bulk-update-dropdown .dropdown-item {
            padding: 0.75rem 1rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .bulk-update-dropdown .dropdown-item:last-child {
            border-bottom: none;
        }
        
        .bulk-update-dropdown .dropdown-item:hover {
            background-color: #f8f9fa;
            transform: translateX(5px);
            transition: all 0.2s ease;
        }
        
        .bulk-update-info {
            font-size: 0.85rem;
            color: #6c757d;
            margin-top: 0.25rem;
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <!-- Main Content -->
        <div class="col-md-10 ms-sm-auto px-md-4">
            <?php include '../includes/topbar.php'; ?>
            <div class="container-fluid mt-4">
                <!-- Page Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="mb-0">Products Management</h2>
                    <div class="d-flex gap-2">
                        <a href="stockreport.php" class="btn btn-info">
                            <i class="fas fa-chart-bar"></i> Stock Report
                        </a>
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#categoryModal" data-permission="manage_categories" data-permission-label="Manage Categories">
                            <i class="fas fa-tags"></i> Manage Categories
                        </button>
                        <a href="adjustment_history.php" class="btn btn-info">
                            <i class="fas fa-history"></i> Stock Adjust History
                        </a>
                        <button type="button" class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addProductNameModal" data-permission="add_product" data-permission-label="Add Product">
                            <i class="fas fa-plus"></i> Add Product Name
                        </button>
                        <button class="btn btn-warning" type="button" data-bs-toggle="modal" data-bs-target="#manualAdjustmentModal" data-permission="adjust_stock" data-permission-label="Adjust Stock">
                            <i class="fas fa-edit"></i>  Add Stock
                        </button>

                    </div>
                </div>

                <!-- Product Name Search Results -->
                <div id="searchResults" class="position-absolute bg-white shadow-sm rounded p-2" style="display: none; z-index: 1000; max-height: 300px; overflow-y: auto; width: 300px;"></div>

                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card stat-card primary">
                            <div class="card-body">
                                <div class="stat-label">Total Products</div>
                                <div class="stat-value"><?php echo number_format($total_products); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-primary" style="width: 100%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card success">
                            <div class="card-body">
                                <div class="stat-label">Total Value</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($total_value, 2); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-success" style="width: 100%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card warning">
                            <div class="card-body">
                                <div class="stat-label">Low Stock</div>
                                <div class="stat-value"><?php echo number_format($low_stock); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-warning" style="width: <?php echo ($total_products > 0) ? ($low_stock / $total_products * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card info">
                            <div class="card-body">
                                <div class="stat-label">Out of Stock</div>
                                <div class="stat-value"><?php echo number_format($out_of_stock); ?></div>
                                <div class="progress mt-2">
                                    <div class="progress-bar bg-info" style="width: <?php echo ($total_products > 0) ? ($out_of_stock / $total_products * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Category</label>
                                <select name="category_id" class="form-select">
                                    <option value="">All Categories</option>
                                    <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>" <?php echo $category_id == $category['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Search</label>
                                <input type="text" name="search" class="form-control" placeholder="Live search: name, category, description, serial numbers..." value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Sort By</label>
                                <select name="sort" class="form-select">
                                    <option value="name" <?php echo $sort === 'name' ? 'selected' : ''; ?>>Name</option>
                                    <option value="price" <?php echo $sort === 'price' ? 'selected' : ''; ?>>Price</option>
                                    <option value="cost_price" <?php echo $sort === 'cost_price' ? 'selected' : ''; ?>>Cost</option>
                                    <option value="quantity" <?php echo $sort === 'quantity' ? 'selected' : ''; ?>>Quantity</option>
                                    <option value="times_sold" <?php echo $sort === 'times_sold' ? 'selected' : ''; ?>>Times Sold</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Order</label>
                                <select name="order" class="form-select">
                                    <option value="asc" <?php echo $order === 'asc' ? 'selected' : ''; ?>>Ascending</option>
                                    <option value="desc" <?php echo $order === 'desc' ? 'selected' : ''; ?>>Descending</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <button type="submit" class="btn btn-primary w-100">Apply Filters</button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Products Table -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">Products List</h5>
                        <span class="badge bg-primary"><?php echo count($products); ?> Products</span>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Product Name</th>
                                    <th>Serial Numbers</th>
                                    <th>Category</th>
                                    <th>Price</th>
                                    <th>Cost</th>
                                    <th>Quantity</th>
                                    <th>Alert Qty</th>
                                    <th>Times Sold</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($products as $product): ?>
                                <tr data-category-id="<?php echo $product['category_id']; ?>" 
                                    data-description="<?php echo htmlspecialchars($product['description']); ?>"
                                    data-serial-numbers="<?php echo htmlspecialchars($product['serial_numbers'] ?? ''); ?>">
                                    <td><?php echo htmlspecialchars($product['name']); ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-info show-serials" data-id="<?php echo $product['id']; ?>">Show</button>
                                    </td>
                                    <td><?php echo htmlspecialchars($product['category_name'] ?? 'No Category'); ?></td>
                                    <td><?php echo $currency . ' ' . number_format($product['price'], 2); ?></td>
                                    <td><?php echo $currency . ' ' . number_format($product['cost_price'], 2); ?></td>
                                    <td><?php echo $product['quantity']; ?></td>
                                    <td><?php echo $product['alert_quantity']; ?></td>
                                    <td><?php echo $product['times_sold']; ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-primary edit-product" data-id="<?php echo $product['id']; ?>" data-permission="edit_product" data-permission-label="Edit Product">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="btn btn-sm btn-danger delete-product" data-id="<?php echo $product['id']; ?>" data-permission="delete_product" data-permission-label="Delete Product">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Product Modal -->
<div class="modal fade" id="addProductModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title">
                    <i class="fas fa-plus me-2"></i>Add New Product
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
            <form method="POST" action="" class="needs-validation" novalidate>
                <div class="modal-body">
                    <!-- Hidden fields to preserve filter state -->
                    <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
                    <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
                    <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
                    <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
                    
                    <div class="mb-3">
                        <label class="form-label">Product Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control" required>
                        <div class="invalid-feedback">Please enter a product name.</div>
                        </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea name="description" class="form-control" rows="2"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Category <span class="text-danger">*</span></label>
                        <select name="category_id" class="form-select" required>
                            <option value="">Select Category</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <div class="invalid-feedback">Please select a category.</div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Selling Price <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">KSH</span>
                                    <input type="number" name="price" class="form-control" step="0.01" min="0" required>
                                </div>
                                <div class="invalid-feedback">Please enter a valid price.</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Cost Price <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">KSH</span>
                                    <input type="number" name="cost_price" class="form-control" step="0.01" min="0" required>
                            </div>
                                <div class="invalid-feedback">Please enter a valid cost price.</div>
                        </div>
                    </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Quantity <span class="text-danger">*</span></label>
                                <input type="number" name="quantity" class="form-control" min="0" required>
                                <div class="invalid-feedback">Please enter a valid quantity.</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Alert Quantity <span class="text-danger">*</span></label>
                                <input type="number" name="alert_quantity" class="form-control" min="0" required>
                                <div class="invalid-feedback">Please enter a valid alert quantity.</div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-2"></i>Cancel
                    </button>
                    <button type="submit" name="add_product" class="btn btn-primary" data-permission="add_product" data-permission-label="Add Product">
                        <i class="fas fa-save me-2"></i>Add Product
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Product Modal -->
<div class="modal fade" id="editProductModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title">
                    <i class="fas fa-edit me-2"></i>Edit Product
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
            <form method="POST" action="" class="needs-validation" novalidate>
                <div class="modal-body">
                    <input type="hidden" name="id" id="edit_id">
                    <!-- Hidden fields to preserve filter state -->
                    <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
                    <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
                    <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
                    <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
                    
                    <div class="mb-3">
                        <label class="form-label">Product Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" id="edit_name" class="form-control" required>
                        <div class="invalid-feedback">Please enter a product name.</div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Category <span class="text-danger">*</span></label>
                        <select name="category_id" id="edit_category" class="form-select" required>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>" 
                                        data-require-serials="<?php echo isset($categories_by_id[$category['id']]) ? (int)$categories_by_id[$category['id']]['require_serials'] : 0; ?>">
                                    <?php echo htmlspecialchars($category['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <div class="invalid-feedback">Please select a category.</div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea name="description" id="edit_description" class="form-control" rows="2"></textarea>
                    </div>

                    <div class="row g-2">
                        <div class="col-md-6">
                <div class="mb-3">
                                <label class="form-label">Selling Price <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">KSH</span>
                                    <input type="number" name="price" id="edit_price" class="form-control" step="0.01" min="0" required>
                </div>
                                <div class="invalid-feedback">Please enter a valid price.</div>
    </div>
</div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Cost Price <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <span class="input-group-text">KSH</span>
                                    <input type="number" name="cost_price" id="edit_cost_price" class="form-control" step="0.01" min="0" required>
                                    <button type="button" class="btn btn-outline-secondary" id="fetchSupplierPrice" title="Get latest supplier price">
                                        <i class="fas fa-sync-alt"></i>
                                    </button>
                                </div>
                                <div class="invalid-feedback">Please enter a valid cost price.</div>
                                <small class="form-text text-muted" id="supplierPriceInfo"></small>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Quantity <span class="text-danger">*</span></label>
                                <input type="number" name="quantity" id="edit_quantity" class="form-control" min="0" required>
                                <div class="invalid-feedback">Please enter a valid quantity.</div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Alert Quantity <span class="text-danger">*</span></label>
                                <input type="number" name="alert_quantity" id="edit_alert_quantity" class="form-control" min="0" required>
                                <div class="invalid-feedback">Please enter a valid alert quantity.</div>
                            </div>
                        </div>
                            </div>
                    
                            <div class="mb-3" id="edit_serial_section" style="display: none;">
                                <label class="form-label">Serial Numbers</label>
                                <div id="edit_serial_wrapper"></div>
                                <small class="form-text text-muted">Enter serial numbers separated by commas or new lines</small>
                            </div>
                            <div class="mb-3" id="edit_no_serial_message" style="display: none;">
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle"></i> This product's category does not require serial numbers. To enable serial numbers, please update the category settings in Manage Categories.
                                </div>
                            </div>
                        </div>
                
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-2"></i>Cancel
                    </button>
                    <button type="submit" name="edit_product" class="btn btn-primary" data-permission="edit_product" data-permission-label="Edit Product">
                        <i class="fas fa-save me-2"></i>Save Changes
                    </button>
      </div>
            </form>
    </div>
  </div>
</div>

<!-- Add Product Name Modal -->
<div class="modal fade" id="addProductNameModal" tabindex="-1" aria-labelledby="addProductNameModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Product Name</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <!-- Hidden fields to preserve filter state -->
                    <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
                    <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
                    <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
                    <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
                    
                    <div class="mb-3">
                        <label class="form-label">Product Name</label>
                        <input type="text" name="name" class="form-control" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea name="description" class="form-control" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Category</label>
                        <select name="category_id" class="form-select" required>
                            <option value="">Select Category</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_product_name" class="btn btn-primary" data-permission="add_product" data-permission-label="Add Product">Add Product Name</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php if (!empty($_SESSION['adjust_success'])): ?>
  <div id="adjustSuccessAlert" class="alert alert-success" style="display:block; position:fixed; top:20px; right:20px; z-index:9999;">
    Stock adjusted successfully.
  </div>
  <script>
    setTimeout(function() {
      var alertBox = document.getElementById('adjustSuccessAlert');
      if (alertBox) alertBox.style.display = 'none';
    }, 3000);
  </script>
  <?php unset($_SESSION['adjust_success']); ?>
<?php endif; ?>

<!-- Add Stock Modal -->
<div class="modal fade" id="addStockModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <div class="modal-header">
                <h5 class="modal-title">Add Stock from Supplier</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Search Supplier</label>
                    <input type="text" class="form-control" id="supplierSearch">
      </div>
                <div id="supplierResults" class="list-group mb-3"></div>
                <div id="supplierItems" class="d-none">
                    <h6>Available Items</h6>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Item</th>
                                    <th>Available</th>
                                    <th>Quantity</th>
                                    <th>Unit Price</th>
                                    <th>Total</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody id="supplierItemsList"></tbody>
                        </table>
                    </div>
                </div>
      </div>
    </div>
  </div>
</div>

<!-- Success Modal -->
<div class="modal fade" id="successModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Success</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
  </div>
            <div class="modal-body">
                <div class="text-center">
                    <i class="fas fa-check-circle text-success fa-3x mb-3"></i>
                    <p id="successMessage"></p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Error Modal -->
<div class="modal fade" id="errorModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title">
                    <i class="fas fa-exclamation-triangle me-2"></i>Error
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="text-center">
                    <i class="fas fa-exclamation-circle text-danger fa-3x mb-3"></i>
                    <div id="errorMessage"></div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Add All Confirmation Modal -->
<div class="modal fade" id="addAllConfirmationModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Add All Items</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to add all selected items from <strong id="confirmSupplierName"></strong> to stock?</p>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> This will add all available quantities of the selected items to your stock.
                </div>
                <div class="progress mt-3" style="display: none;">
                    <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmAddAll">Yes, Add</button>
            </div>
        </div>
    </div>
</div>

<!-- Manual Serial Numbers Modal -->
<div class="modal fade" id="manualSerialsModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Enter Serial Numbers</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Quantity: <span id="manualQuantityDisplay">0</span></label>
                </div>
                <div id="serialNumberInputs">
                    <!-- Dynamic input fields will be added here -->
                </div>
                <div id="manualSerialError" class="text-danger mt-2" style="display: none;">
                    Please fill in all serial numbers.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveManualSerials">Save</button>
            </div>
        </div>
    </div>
</div>

<!-- Category Management Modal -->
<div class="modal fade" id="categoryModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-tags me-3"></i>Manage Categories
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <!-- Add Category Form Section -->
                <div class="category-form-section">
                    <h6><i class="fas fa-plus-circle text-primary"></i>Add New Category</h6>
                    <form method="POST">
                    <!-- Hidden fields to preserve filter state -->
                    <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
                    <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
                    <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
                    <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
                    
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-2">
                                    <label class="form-label">
                                        <i class="fas fa-tag me-2"></i>Category Name
                                    </label>
                                    <input type="text" name="category_name" class="form-control" placeholder="Enter category name" required>
                    </div>
                    </div>
                            <div class="col-md-6">
                                <div class="mb-2">
                                    <label class="form-label">
                                        <i class="fas fa-barcode me-2"></i>Require Serial Numbers
                                    </label>
                        <select class="form-select" name="require_serials" id="require_serials">
                            <option value="0">No</option>
                            <option value="1">Yes</option>
                        </select>
                    </div>
                            </div>
                        </div>
                        <div class="mb-2">
                            <label class="form-label">
                                <i class="fas fa-align-left me-2"></i>Description
                            </label>
                            <textarea name="category_description" class="form-control" rows="2" placeholder="Enter category description (optional)"></textarea>
                        </div>
                        <div class="text-end">
                            <button type="submit" name="add_category" class="btn btn-primary" data-permission="manage_categories" data-permission-label="Manage Categories">
                                <i class="fas fa-plus me-2"></i>Add Category
                            </button>
                        </div>
                </form>
                </div>

                <!-- Categories List Section -->
                <div class="categories-list-section">
                    <h6><i class="fas fa-list text-info"></i>Existing Categories</h6>
                    <?php if (empty($categories)): ?>
                        <div class="categories-empty-state">
                            <i class="fas fa-folder-open"></i>
                            <p class="mb-0">No categories found. Add your first category above!</p>
                        </div>
                    <?php else: ?>
                        <div class="categories-table-container">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                        <th><i class="fas fa-tag me-2"></i>Name</th>
                                        <th><i class="fas fa-align-left me-2"></i>Description</th>
                                        <th><i class="fas fa-barcode me-2"></i>Serial Req.</th>
                                        <th><i class="fas fa-cogs me-2"></i>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($categories as $category): ?>
                            <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($category['name']); ?></strong>
                                        </td>
                                        <td>
                                            <span class="text-muted">
                                                <?php echo !empty($category['description']) ? htmlspecialchars($category['description']) : '<em>No description</em>'; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (isset($categories_by_id[$category['id']]) && $categories_by_id[$category['id']]['require_serials']): ?>
                                                <span class="badge bg-success">
                                                    <i class="fas fa-check me-1"></i>Yes
                                                </span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">
                                                    <i class="fas fa-times me-1"></i>No
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group" role="group">
                                    <button class="btn btn-sm btn-primary edit-category" 
                                            data-id="<?php echo $category['id']; ?>"
                                            data-name="<?php echo htmlspecialchars($category['name']); ?>"
                                            data-description="<?php echo htmlspecialchars($category['description'] ?? ''); ?>"
                                            data-require-serials="<?php echo isset($categories_by_id[$category['id']]) ? (int)$categories_by_id[$category['id']]['require_serials'] : 0; ?>"
                                                        data-permission="manage_categories" data-permission-label="Manage Categories"
                                                        title="Edit Category">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                                <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this category? This action cannot be undone.');">
                                        <input type="hidden" name="category_id" value="<?php echo $category['id']; ?>">
                                        <!-- Hidden fields to preserve filter state -->
                                        <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
                                        <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
                                        <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
                                        <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
                                        
                                                    <button type="submit" name="delete_category" class="btn btn-sm btn-danger" 
                                                            data-permission="manage_categories" data-permission-label="Manage Categories"
                                                            title="Delete Category">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </form>
                                            </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                        <div class="mt-2">
                            <small class="text-muted">
                                <i class="fas fa-info-circle me-1"></i>
                                Total: <strong><?php echo count($categories); ?></strong> categories
                            </small>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Manual Stock Adjustment Modal -->
<div class="modal fade" id="manualAdjustmentModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Manually Update Stock - Multiple Products</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" id="stockAdjustmentForm">
                <div class="modal-body">
                    <!-- Hidden fields to preserve filter state -->
                    <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
                    <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
                    <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
                    <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
                    
                    <div id="stockAdjustmentAlerts"></div>
                    <div class="mb-3">
                        <label class="form-label">Supplier</label>
                        <select name="supplier_id" class="form-select" required>
                            <option value="">Select Supplier</option>
                            <?php foreach ($suppliers as $supplier): ?>
                                <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Products</label>
                        <div id="productsContainer" style="max-height: 400px; overflow-y: auto; padding-right: 10px;">
                            <!-- Product entries will be added here -->
                        </div>
                        <button type="button" class="btn btn-outline-primary btn-sm mt-2" id="addProductBtn">
                            <i class="fas fa-plus"></i> Add Product
                        </button>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="adjust_stock_multiple" class="btn btn-primary">Update All Products</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Confirmation Modal for Stock Update -->
<div class="modal fade" id="confirmStockUpdateModal" tabindex="-1" aria-labelledby="confirmStockUpdateLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="confirmStockUpdateLabel">Confirm Stock Update</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        Are you sure you want to update the stock for this product?
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" id="confirmStockUpdateBtn" class="btn btn-primary">Yes, Update Stock</button>
      </div>
    </div>
  </div>
</div>

<!-- Success Modal for Stock Update -->
<div class="modal fade" id="successStockUpdateModal" tabindex="-1" aria-labelledby="successStockUpdateLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="successStockUpdateLabel">Stock Updated</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body text-success text-center">
        <i class="fas fa-check-circle fa-2x mb-2"></i><br>
        Stock has been updated successfully!
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-success" data-bs-dismiss="modal">OK</button>
      </div>
    </div>
  </div>
</div>

<!-- Delete Product Modal -->
<div class="modal fade" id="deleteProductModal" tabindex="-1" aria-labelledby="deleteProductModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header bg-danger text-white">
        <h5 class="modal-title" id="deleteProductModalLabel"><i class="fas fa-exclamation-triangle me-2"></i>Delete Product</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div id="deleteProductModalMsg" class="mb-2"></div>
        <p>Are you sure you want to delete this product? This action cannot be undone.</p>
      </div>
      <div class="modal-footer bg-light">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <form id="deleteProductForm" method="POST" style="display:inline;">
          <input type="hidden" name="id" id="delete_product_id">
          <input type="hidden" name="delete_product" value="1">
          <!-- Hidden fields to preserve filter state -->
          <input type="hidden" name="filter_category_id" value="<?php echo htmlspecialchars($category_id); ?>">
          <input type="hidden" name="filter_search" value="<?php echo htmlspecialchars($search); ?>">
          <input type="hidden" name="filter_sort" value="<?php echo htmlspecialchars($sort); ?>">
          <input type="hidden" name="filter_order" value="<?php echo htmlspecialchars($order); ?>">
          
          <button type="submit" class="btn btn-danger" data-permission="delete_product" data-permission-label="Delete Product">Delete</button>
        </form>
      </div>
    </div>
  </div>
</div>

<!-- Edit Confirmation Modal -->
<div class="modal fade" id="editConfirmModal" tabindex="-1" aria-labelledby="editConfirmModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header bg-warning text-dark">
        <h5 class="modal-title" id="editConfirmModalLabel"><i class="fas fa-exclamation-triangle me-2"></i>Confirm Edit</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="mb-2">You are about to update this product's details. Please confirm you want to proceed. <br><b>Warning:</b> Editing product details may affect inventory and sales records.</div>
      </div>
      <div class="modal-footer bg-light">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-warning" id="confirmEditBtn">Yes, Save Changes</button>
      </div>
    </div>
  </div>
</div>

<!-- Serial Numbers Modal -->
<div class="modal fade" id="serialNumbersModal" tabindex="-1" aria-labelledby="serialNumbersModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header bg-primary text-white">
        <h5 class="modal-title" id="serialNumbersModalLabel">Available Serial Numbers</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" id="serialNumbersModalBody">
        <!-- Serial numbers will be loaded here -->
      </div>
      <div class="modal-footer bg-light">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
        </div>
    </div>
</div>

<!-- Add the confirmation modal HTML -->
<div id="confirmUpdateStockModal" class="modal">
    <div class="modal-content">
        <h2>Confirm Stock Update</h2>
        <p>Are you sure you want to update the stock for this product?</p>
        <button id="confirmUpdateStock">Yes, Update Stock</button>
        <button id="cancelUpdateStock">Cancel</button>
    </div>
</div>



<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Form validation
    const forms = document.querySelectorAll('.needs-validation');
    Array.from(forms).forEach(form => {
        form.addEventListener('submit', event => {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            form.classList.add('was-validated');
        }, false);
    });

    // Function to handle edit button click
    window.editProduct = function(id) {
        console.log('Edit button clicked for product:', id);
        
        // Use correct path for API endpoints
        fetch('../api/get_product.php?id=' + id)
            .then(response => {
                console.log('Response status:', response.status);
                return response.text().then(text => {
                    try {
                        const data = JSON.parse(text);
                        if (!response.ok) {
                            throw new Error(data.message || data.error || 'Failed to fetch product data');
                        }
                        return data;
                    } catch (e) {
                        console.error('Response text:', text);
                        throw new Error('Invalid JSON response from server');
                    }
                });
            })
            .then(data => {
                console.log('Product data:', data);
                
                if (!data || typeof data !== 'object') {
                    throw new Error('Invalid product data received');
                }
                
                // Update form fields
                const fields = {
                    'edit_id': data.id,
                    'edit_name': data.name,
                    'edit_category': data.category_id,
                    'edit_price': data.price,
                    'edit_cost_price': data.cost_price,
                    'edit_quantity': data.quantity,
                    'edit_alert_quantity': data.alert_quantity,
                    'edit_description': data.description
                };
                
                // Update each field
                Object.entries(fields).forEach(([id, value]) => {
                    const element = document.getElementById(id);
                    if (element) {
                        element.value = value;
                    } else {
                        console.warn(`Element not found: ${id}`);
                    }
                });

                // Check if category requires serial numbers
                const categorySelect = document.getElementById('edit_category');
                const selectedOption = categorySelect.options[categorySelect.selectedIndex];
                const requiresSerials = selectedOption.getAttribute('data-require-serials') === '1';
                
                const serialSection = document.getElementById('edit_serial_section');
                const noSerialMessage = document.getElementById('edit_no_serial_message');
                
                if (requiresSerials) {
                    serialSection.style.display = 'block';
                    noSerialMessage.style.display = 'none';
                    
                    // Get serial numbers for the product
                    return fetch('../api/get_serial_numbers.php?product_id=' + id)
                        .then(response => response.json())
                        .then(serialData => {
                            let serials = serialData.serial_numbers || [];
                            let serialInput = document.getElementById('edit_serial');
                            if (!serialInput) {
                                const serialWrapper = document.getElementById('edit_serial_wrapper');
                                serialWrapper.innerHTML = '';
                                serialInput = document.createElement('textarea');
                                serialInput.name = 'serial_numbers';
                                serialInput.id = 'edit_serial';
                                serialInput.className = 'form-control';
                                serialInput.rows = 3;
                                serialWrapper.appendChild(serialInput);
                                serialInput.addEventListener('input', validateSerialNumbers);
                            }
                            serialInput.value = serials.join('\n');
                        });
                } else {
                    serialSection.style.display = 'none';
                    noSerialMessage.style.display = 'block';
                }
            })
            .then(() => {
                // Show the modal
                const editModal = new bootstrap.Modal(document.getElementById('editProductModal'));
                editModal.show();
            })
            .catch(error => {
                console.error('Error details:', error);
                console.error('Error stack:', error.stack);
                alert('Error: ' + error.message);
            });
    };

    // Function to handle delete button click
    window.deleteProduct = function(id) {
        if (confirm('Are you sure you want to delete this product?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';

            const idInput = document.createElement('input');
            idInput.type = 'hidden';
            idInput.name = 'id';
            idInput.value = id;

            const submitInput = document.createElement('input');
            submitInput.type = 'hidden';
            submitInput.name = 'delete_product';
            submitInput.value = '1';

            // Add filter preservation fields from current page state
            const categorySelect = document.querySelector('select[name="category_id"]');
            const searchInput = document.querySelector('input[name="search"]');
            const sortSelect = document.querySelector('select[name="sort"]');
            const orderSelect = document.querySelector('select[name="order"]');

            const filterCategoryInput = document.createElement('input');
            filterCategoryInput.type = 'hidden';
            filterCategoryInput.name = 'filter_category_id';
            filterCategoryInput.value = categorySelect ? categorySelect.value : '';

            const filterSearchInput = document.createElement('input');
            filterSearchInput.type = 'hidden';
            filterSearchInput.name = 'filter_search';
            filterSearchInput.value = searchInput ? searchInput.value : '';

            const filterSortInput = document.createElement('input');
            filterSortInput.type = 'hidden';
            filterSortInput.name = 'filter_sort';
            filterSortInput.value = sortSelect ? sortSelect.value : '';

            const filterOrderInput = document.createElement('input');
            filterOrderInput.type = 'hidden';
            filterOrderInput.name = 'filter_order';
            filterOrderInput.value = orderSelect ? orderSelect.value : '';

            form.appendChild(idInput);
            form.appendChild(submitInput);
            form.appendChild(filterCategoryInput);
            form.appendChild(filterSearchInput);
            form.appendChild(filterSortInput);
            form.appendChild(filterOrderInput);
            document.body.appendChild(form);
            form.submit();
        }
    };

    // Category Management with Beautiful UX
    const editCategoryButtons = document.querySelectorAll('.edit-category');
    const categoryModal = document.getElementById('categoryModal');
    
    // Enhanced edit category functionality
    editCategoryButtons.forEach(button => {
        button.addEventListener('click', function() {
            const id = this.dataset.id;
            const name = this.dataset.name;
            const description = this.dataset.description;
            const requireSerials = this.dataset.requireSerials || '0';
            
            // Create a beautiful edit modal instead of inline editing
            const editModal = document.createElement('div');
            editModal.className = 'modal fade';
            editModal.id = 'editCategoryModal';
            editModal.tabIndex = -1;
            editModal.innerHTML = `
                <div class="modal-dialog">
                    <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
                        <div class="modal-header" style="background: linear-gradient(135deg, #764ba2 0%, #667eea 100%); border: none; color: white; padding: 20px 30px;">
                            <h5 class="modal-title" style="font-weight: 700;">
                                <i class="fas fa-edit me-3"></i>Edit Category
                            </h5>
                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                        </div>
                        <form method="POST">
                            <div class="modal-body" style="padding: 30px; background: #fafbfc;">
                <input type="hidden" name="category_id" value="${id}">
                                <input type="hidden" name="filter_category_id" value="${document.querySelector('select[name="category_id"]')?.value || ''}">
                                <input type="hidden" name="filter_search" value="${document.querySelector('input[name="search"]')?.value || ''}">
                                <input type="hidden" name="filter_sort" value="${document.querySelector('select[name="sort"]')?.value || ''}">
                                <input type="hidden" name="filter_order" value="${document.querySelector('select[name="order"]')?.value || ''}">
                                
                                <div class="row">
                                    <div class="col-md-8">
                <div class="mb-3">
                                            <label class="form-label" style="font-weight: 600; color: #495057;">
                                                <i class="fas fa-tag me-2"></i>Category Name
                                            </label>
                                            <input type="text" name="category_name" class="form-control" value="${name}" required 
                                                   style="border: 2px solid #e9ecef; border-radius: 12px; padding: 12px 16px;">
                </div>
                </div>
                                    <div class="col-md-4">
                <div class="mb-3">
                                            <label class="form-label" style="font-weight: 600; color: #495057;">
                                                <i class="fas fa-barcode me-2"></i>Serial Numbers
                                            </label>
                                            <select class="form-select" name="require_serials" 
                                                    style="border: 2px solid #e9ecef; border-radius: 12px; padding: 12px 16px;">
                        <option value="0" ${requireSerials === '0' ? 'selected' : ''}>No</option>
                        <option value="1" ${requireSerials === '1' ? 'selected' : ''}>Yes</option>
                    </select>
                </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label" style="font-weight: 600; color: #495057;">
                                        <i class="fas fa-align-left me-2"></i>Description
                                    </label>
                                    <textarea name="category_description" class="form-control" rows="3" 
                                              style="border: 2px solid #e9ecef; border-radius: 12px; padding: 12px 16px;">${description}</textarea>
                                </div>
                            </div>
                            <div class="modal-footer" style="border: none; padding: 20px 30px; background: #fafbfc;">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" 
                                        style="border-radius: 12px; padding: 12px 24px;">Cancel</button>
                                <button type="submit" name="edit_category" class="btn btn-primary" 
                                        style="background: linear-gradient(135deg, #764ba2 0%, #667eea 100%); border: none; border-radius: 12px; padding: 12px 24px; font-weight: 600; box-shadow: 0 4px 12px rgba(118,75,162,0.3);"
                                        data-permission="manage_categories" data-permission-label="Manage Categories">
                                    <i class="fas fa-save me-2"></i>Save Changes
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            `;
            
            // Add modal to body and show it
            document.body.appendChild(editModal);
            const modal = new bootstrap.Modal(editModal);
            modal.show();
            
            // Clean up modal after hiding
            editModal.addEventListener('hidden.bs.modal', function() {
                document.body.removeChild(editModal);
            });
        });
    });
    
    // Add form submission animation
    if (categoryModal) {
        const categoryForm = categoryModal.querySelector('form');
        if (categoryForm) {
            categoryForm.addEventListener('submit', function() {
                const formSection = this.closest('.category-form-section');
                const submitBtn = this.querySelector('button[type="submit"]');
                
                // Add loading state
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Adding Category...';
                submitBtn.disabled = true;
                
                // Add success animation class
                formSection.classList.add('success');
                
                // Remove animation class after animation completes
                setTimeout(() => {
                    formSection.classList.remove('success');
                }, 600);
            });
        }
    }
    
    // Smooth scroll for long category lists
    const categoriesContainer = document.querySelector('.categories-table-container');
    if (categoriesContainer) {
        categoriesContainer.style.scrollBehavior = 'smooth';
    }

    // Manual Stock Adjustment - autofill category, price, cost
    const adjustmentProduct = document.getElementById('adjustmentProduct');
    const adjustmentCategory = document.getElementById('adjustmentCategory');
    const adjustmentUnitPrice = document.getElementById('adjustmentUnitPrice');
    const adjustmentCurrentPrice = document.getElementById('adjustmentCurrentPrice');
    const serialNumbersContainer = document.getElementById('serialNumbersContainer');
    const serialNumbersInput = document.getElementById('serialNumbersInput');
    const adjustmentQuantity = document.getElementById('adjustmentQuantity');

    if (adjustmentProduct) {
        adjustmentProduct.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const category = selectedOption.getAttribute('data-category');
            const price = selectedOption.getAttribute('data-price');
            const requireSerials = selectedOption.getAttribute('data-require-serials');
            
            if (adjustmentCategory) adjustmentCategory.value = category || '';
            if (adjustmentCurrentPrice) adjustmentCurrentPrice.value = price || '';
            if (adjustmentUnitPrice) adjustmentUnitPrice.value = price || '';
            
            // Show/hide serial numbers input based on category requirement
            if (requireSerials === '1') {
                serialNumbersContainer.style.display = '';
                if (serialNumbersInput) {
                    serialNumbersInput.required = true;
                }
            } else {
                serialNumbersContainer.style.display = 'none';
                if (serialNumbersInput) {
                    serialNumbersInput.required = false;
                    serialNumbersInput.value = '';
                }
            }
        });
    }

    // Add validation for serial numbers matching quantity
    const stockAdjustmentForm = document.getElementById('stockAdjustmentForm');
    if (stockAdjustmentForm) {
        stockAdjustmentForm.addEventListener('submit', function(e) {
            const quantity = parseInt(adjustmentQuantity.value) || 0;
            const unitPrice = parseFloat(adjustmentUnitPrice.value) || 0;
            const requireSerials = adjustmentProduct.options[adjustmentProduct.selectedIndex].getAttribute('data-require-serials');
            
            // Validate unit price
            if (unitPrice <= 0) {
                e.preventDefault();
                alert('Please enter a valid unit price greater than 0');
                return;
            }
            
            if (requireSerials === '1' && quantity > 0) {
                const serials = serialNumbersInput.value.split(/[\n,]/).map(s => s.trim()).filter(s => s);
                if (serials.length !== quantity) {
                    e.preventDefault();
                    alert(`Please enter exactly ${quantity} serial number(s)`);
                }
            }
        });
    }

    // Update serial numbers input when quantity changes
    if (adjustmentQuantity) {
        adjustmentQuantity.addEventListener('input', function() {
            const quantity = parseInt(this.value) || 0;
            const requireSerials = adjustmentProduct.options[adjustmentProduct.selectedIndex].getAttribute('data-require-serials');
            
            if (requireSerials === '1' && quantity > 0) {
                serialNumbersInput.placeholder = `Enter ${quantity} serial number(s), one per line or separated by commas`;
            }
        });
    }

    const showConfirmStockUpdate = document.getElementById('showConfirmStockUpdate');
    const confirmStockUpdateModal = new bootstrap.Modal(document.getElementById('confirmStockUpdateModal'));
    const confirmStockUpdateBtn = document.getElementById('confirmStockUpdateBtn');

    if (showConfirmStockUpdate && confirmStockUpdateBtn && stockAdjustmentForm) {
        showConfirmStockUpdate.addEventListener('click', function(e) {
            e.preventDefault();
            confirmStockUpdateModal.show();
        });
        confirmStockUpdateBtn.addEventListener('click', function() {
            confirmStockUpdateModal.hide();
            stockAdjustmentForm.submit();
        });
    }

    // Attach event listeners for edit and delete buttons
    document.querySelectorAll('.edit-product').forEach(function(btn) {
        btn.addEventListener('click', function() {
            editProduct(this.getAttribute('data-id'));
        });
    });
    document.querySelectorAll('.delete-product').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var id = this.getAttribute('data-id');
            document.getElementById('delete_product_id').value = id;
            document.getElementById('deleteProductModalMsg').innerHTML = '';
            var modal = new bootstrap.Modal(document.getElementById('deleteProductModal'));
            modal.show();
        });
    });

    // Intercept edit form submission for confirmation
    var editForm = document.querySelector('#editProductModal form');
    var confirmEditBtn = document.getElementById('confirmEditBtn');
    var confirmModalEl = document.getElementById('editConfirmModal');
    var confirmModal = confirmModalEl ? new bootstrap.Modal(confirmModalEl) : null;

    if (editForm && confirmEditBtn && confirmModal) {
        editForm.addEventListener('submit', function(e) {
            if (!editForm.classList.contains('edit-confirmed')) {
                e.preventDefault();
                // Hide the edit modal if open
                var editModal = bootstrap.Modal.getInstance(document.getElementById('editProductModal'));
                if (editModal) editModal.hide();
                confirmModal.show();
            } else {
                // Remove the flag for future edits
                editForm.classList.remove('edit-confirmed');
            }
        });

        confirmEditBtn.onclick = function() {
            editForm.classList.add('edit-confirmed');
            confirmModal.hide();
            setTimeout(function() {
                // Find the submit button with name="edit_product"
                var submitBtn = editForm.querySelector('button[name="edit_product"]');
                if (submitBtn) {
                    submitBtn.click();
                } else if (editForm.requestSubmit) {
                    editForm.requestSubmit();
                } else {
                    editForm.submit();
                }
            }, 300);
        };
    }

    document.querySelectorAll('.show-serials').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var productId = this.getAttribute('data-id');
            var modalBody = document.getElementById('serialNumbersModalBody');
            modalBody.innerHTML = '<div class="text-center text-muted">Loading...</div>';
            var modal = new bootstrap.Modal(document.getElementById('serialNumbersModal'));
            modal.show();
            fetch('get_available_serials.php?product_id=' + productId)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        modalBody.innerHTML = '<ul class="list-group">' +
                            data.serial_numbers.map(s => '<li class="list-group-item">' + s + '</li>').join('') +
                            '</ul>';
                    } else {
                        modalBody.innerHTML = '<div class="text-danger text-center">' + data.message + '</div>';
                    }
                })
                .catch(() => {
                    modalBody.innerHTML = '<div class="text-danger text-center">Error loading serial numbers.</div>';
                });
        });
    });

    // Add event listener for category change in edit modal
    document.getElementById('edit_category').addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const requiresSerials = selectedOption.getAttribute('data-require-serials') === '1';
        
        const serialSection = document.getElementById('edit_serial_section');
        const noSerialMessage = document.getElementById('edit_no_serial_message');
        
        if (requiresSerials) {
            serialSection.style.display = 'block';
            noSerialMessage.style.display = 'none';
        } else {
            serialSection.style.display = 'none';
            noSerialMessage.style.display = 'block';
        }
    });

    // Update the JavaScript to show the confirmation modal
    const updateStockBtn = document.getElementById('updateStockBtn');
    const confirmUpdateStock = document.getElementById('confirmUpdateStock');
    const cancelUpdateStock = document.getElementById('cancelUpdateStock');
    const confirmUpdateStockModal = document.getElementById('confirmUpdateStockModal');

    if (updateStockBtn && confirmUpdateStockModal) {
        updateStockBtn.addEventListener('click', function() {
            confirmUpdateStockModal.style.display = 'block';
        });
    }

    if (confirmUpdateStock && confirmUpdateStockModal) {
        confirmUpdateStock.addEventListener('click', function() {
            // Proceed with stock update logic
            confirmUpdateStockModal.style.display = 'none';
            // Add your stock update logic here
        });
    }

    if (cancelUpdateStock && confirmUpdateStockModal) {
        cancelUpdateStock.addEventListener('click', function() {
            confirmUpdateStockModal.style.display = 'none';
        });
    }

    // Add validation for serial numbers matching quantity in edit modal
    const editProductForm = document.querySelector('#editProductModal form');
    const editQuantity = document.getElementById('edit_quantity');
    let editSerial = document.getElementById('edit_serial'); // let, not const, so we can reassign
    const editCategory = document.getElementById('edit_category');

    function validateSerialNumbers() {
        if (!editQuantity || !editCategory) {
            return true;
        }
        const quantity = parseInt(editQuantity.value) || 0;
        const selectedOption = editCategory.options[editCategory.selectedIndex];
        const requiresSerials = selectedOption.getAttribute('data-require-serials') === '1';
        if (requiresSerials && quantity > 0) {
            // Check if serial input exists
            editSerial = document.getElementById('edit_serial'); // update reference in case it was created dynamically
            if (!editSerial) {
                // Don't warn, just skip validation
                return true;
            }
            const serials = editSerial.value.split(/[\n,]/).map(s => s.trim()).filter(s => s);
            if (serials.length !== quantity) {
                editQuantity.setCustomValidity(`Please enter exactly ${quantity} serial number(s)`);
                return false;
            }
        }
        editQuantity.setCustomValidity('');
        return true;
    }

    if (editProductForm) {
        editProductForm.addEventListener('submit', function(e) {
            if (!validateSerialNumbers()) {
                e.preventDefault();
                alert('Please ensure the number of serial numbers matches the quantity.');
            }
        });
    }

    if (editQuantity) {
        editQuantity.addEventListener('input', function() {
            validateSerialNumbers();
        });
    }

    // Remove the global event listener for editSerial here
    // Instead, attach it dynamically when the textarea is created

    if (editCategory) {
        editCategory.addEventListener('change', function() {
            validateSerialNumbers();
        });
    }

    // Patch the code that creates the serial textarea to add the event listener
    // Find: let serialInput = document.getElementById('edit_serial');
    // After serialInput is created, add:
    // serialInput.addEventListener('input', validateSerialNumbers);

    // Initialize with one product entry when modal opens


    // Supplier Price Fetching Functionality
    const fetchSupplierPriceBtn = document.getElementById('fetchSupplierPrice');
    const supplierPriceInfo = document.getElementById('supplierPriceInfo');
    
    if (fetchSupplierPriceBtn) {
        fetchSupplierPriceBtn.addEventListener('click', function() {
            const productId = document.getElementById('edit_id').value;
            const costPriceInput = document.getElementById('edit_cost_price');
            
            if (!productId) {
                alert('Please select a product first.');
                return;
            }
            
            // Show loading state
            fetchSupplierPriceBtn.disabled = true;
            fetchSupplierPriceBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            supplierPriceInfo.innerHTML = 'Fetching supplier price...';
            
            // Fetch supplier price from API
            fetch(`../api/get_supplier_price.php?product_id=${productId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        throw new Error(data.error);
                    }
                    
                    if (data.unit_price !== null) {
                        // Update cost price field
                        costPriceInput.value = data.unit_price;
                        
                        // Show success message with supplier info
                        const supplierName = data.supplier_name || 'Unknown Supplier';
                        const date = new Date(data.created_at).toLocaleDateString();
                        supplierPriceInfo.innerHTML = `<i class="fas fa-check text-success"></i> Latest supplier price: KSH ${data.unit_price} (from ${supplierName} on ${date})`;
                        supplierPriceInfo.className = 'form-text text-success';
                    } else {
                        // No supplier receipts found
                        supplierPriceInfo.innerHTML = `<i class="fas fa-info text-info"></i> ${data.message}`;
                        supplierPriceInfo.className = 'form-text text-info';
                    }
                })
                .catch(error => {
                    console.error('Error fetching supplier price:', error);
                    supplierPriceInfo.innerHTML = `<i class="fas fa-exclamation-triangle text-danger"></i> Error: ${error.message}`;
                    supplierPriceInfo.className = 'form-text text-danger';
                })
                .finally(() => {
                    // Reset button state
                    fetchSupplierPriceBtn.disabled = false;
                    fetchSupplierPriceBtn.innerHTML = '<i class="fas fa-sync-alt"></i>';
                });
        });
    }
});
</script>

<?php if (!empty($error) || !empty($success)): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var msg = '';
    <?php if (!empty($error)): ?>
        msg = '<?php echo htmlspecialchars($error); ?>';
        document.getElementById('errorMessage').innerHTML = msg;
        var modal = new bootstrap.Modal(document.getElementById('errorModal'));
        modal.show();
    <?php elseif (!empty($success)): ?>
        msg = '<?php echo htmlspecialchars($success); ?>';
        document.getElementById('successMessage').innerHTML = msg;
        var modal = new bootstrap.Modal(document.getElementById('successModal'));
        modal.show();
    <?php endif; ?>
});
</script>
<?php endif; ?>

<?php if (!empty($success)): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // After success, reload to avoid resubmission
    setTimeout(function() {
        window.location.href = 'products.php';
    }, 1200);
});
</script>
<?php endif; ?>

<!-- Multiple Product Stock Adjustment -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Multiple Product Stock Adjustment
    const productsContainer = document.getElementById('productsContainer');
    const addProductBtn = document.getElementById('addProductBtn');
    const stockAdjustmentForm = document.getElementById('stockAdjustmentForm');
    const manualAdjustmentModal = document.getElementById('manualAdjustmentModal');
    let productCounter = 0;

    // Function to create a product entry
    function createProductEntry() {
        productCounter++;
        const productEntry = document.createElement('div');
        productEntry.className = 'product-entry border rounded p-3 mb-3';
        productEntry.innerHTML = `
            <div class="d-flex justify-content-between align-items-start mb-2">
                <h6 class="mb-0">Product ${productCounter}</h6>
                <button type="button" class="btn btn-outline-danger btn-sm remove-product" onclick="removeProduct(this)">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="row">
                <div class="col-md-6 mb-2">
                    <label class="form-label">Product</label>
                    <select name="products[${productCounter}][product_id]" class="form-select product-select" required>
                        <option value="">Select Product</option>
                        <?php foreach ($products_for_dropdown as $product): ?>
                            <option value="<?php echo $product['id']; ?>"
                                data-category="<?php echo $product['category_id']; ?>"
                                data-require-serials="<?php echo $product['require_serials']; ?>"
                                data-price="<?php echo htmlspecialchars($product['price']); ?>"
                                data-current-stock="<?php echo $product['quantity']; ?>">
                                <?php echo htmlspecialchars($product['name']); ?> (Stock: <?php echo $product['quantity']; ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-3 mb-2">
                    <label class="form-label">Quantity</label>
                    <input type="number" name="products[${productCounter}][quantity]" class="form-control quantity-input" required>
                    <small class="form-text text-muted">+ to add, - to remove</small>
                </div>
                <div class="col-md-3 mb-2">
                    <label class="form-label">Unit Price</label>
                    <input type="number" name="products[${productCounter}][unit_price]" class="form-control price-input" step="0.01" min="0" required>
                </div>
            </div>
            <div class="row">
                <div class="col-md-6 mb-2">
                    <label class="form-label">Category</label>
                    <select class="form-select category-display" disabled>
                        <option value="">Select Category</option>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6 mb-2">
                    <label class="form-label">Current Stock</label>
                    <input type="text" class="form-control current-stock-display" readonly>
                </div>
            </div>
            <div class="serial-numbers-section" style="display: none;">
                <label class="form-label">Serial Numbers</label>
                <textarea name="products[${productCounter}][serial_numbers]" class="form-control serial-numbers-input" rows="2" placeholder="Enter serial numbers, one per line or separated by commas"></textarea>
                <small class="form-text text-muted">Enter one serial number per line or separated by commas</small>
            </div>
        `;
        
        // Add event listeners to the new product entry
        const productSelect = productEntry.querySelector('.product-select');
        const quantityInput = productEntry.querySelector('.quantity-input');
        const priceInput = productEntry.querySelector('.price-input');
        const categoryDisplay = productEntry.querySelector('.category-display');
        const currentStockDisplay = productEntry.querySelector('.current-stock-display');
        const serialNumbersSection = productEntry.querySelector('.serial-numbers-section');
        const serialNumbersInput = productEntry.querySelector('.serial-numbers-input');
        
        // Product selection change handler
        productSelect.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const category = selectedOption.getAttribute('data-category');
            const price = selectedOption.getAttribute('data-price');
            const currentStock = selectedOption.getAttribute('data-current-stock');
            const requireSerials = selectedOption.getAttribute('data-require-serials');
            
            categoryDisplay.value = category || '';
            currentStockDisplay.value = currentStock || '';
            priceInput.value = price || '';
            
            // Show/hide serial numbers section
            if (requireSerials === '1') {
                serialNumbersSection.style.display = 'block';
                serialNumbersInput.required = true;
            } else {
                serialNumbersSection.style.display = 'none';
                serialNumbersInput.required = false;
                serialNumbersInput.value = '';
            }
        });
        
        // Quantity change handler for serial numbers
        quantityInput.addEventListener('input', function() {
            const quantity = parseInt(this.value) || 0;
            const requireSerials = productSelect.options[productSelect.selectedIndex]?.getAttribute('data-require-serials');
            
            if (requireSerials === '1' && quantity > 0) {
                serialNumbersInput.placeholder = `Enter ${quantity} serial number(s), one per line or separated by commas`;
            }
        });
        
        return productEntry;
    }

    // Function to remove a product entry
    window.removeProduct = function(button) {
        const productEntry = button.closest('.product-entry');
        productEntry.remove();
        updateProductNumbers();
        updateScrollIndicator();
    };

    // Function to update product numbers
    function updateProductNumbers() {
        const productEntries = document.querySelectorAll('.product-entry');
        productEntries.forEach((entry, index) => {
            const title = entry.querySelector('h6');
            title.textContent = `Product ${index + 1}`;
        });
        productCounter = productEntries.length;
    }

    // Add product button handler
    if (addProductBtn) {
        addProductBtn.addEventListener('click', function() {
            const productEntry = createProductEntry();
            productsContainer.appendChild(productEntry);
            
            // Auto-scroll to the newly added product entry
            setTimeout(() => {
                productEntry.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
            }, 100);
            
            // Add scroll indicator if needed
            updateScrollIndicator();
        });
    }
    
    // Function to update scroll indicator
    function updateScrollIndicator() {
        const productEntries = document.querySelectorAll('.product-entry');
        const scrollIndicator = document.getElementById('scrollIndicator');
        
        if (productEntries.length > 2) {
            if (!scrollIndicator) {
                const indicator = document.createElement('div');
                indicator.id = 'scrollIndicator';
                indicator.className = 'text-muted small text-center mt-2';
                indicator.innerHTML = '<i class="fas fa-arrows-alt-v"></i> Scroll to see more products';
                productsContainer.parentNode.insertBefore(indicator, addProductBtn);
            }
        } else {
            if (scrollIndicator) {
                scrollIndicator.remove();
            }
        }
    }

    // Form validation for multiple products
    if (stockAdjustmentForm) {
        stockAdjustmentForm.addEventListener('submit', function(e) {
            const alertPlaceholder = document.getElementById('stockAdjustmentAlerts');
            alertPlaceholder.innerHTML = ''; // Clear previous alerts
            const productEntries = document.querySelectorAll('.product-entry');
            let isValid = true;
            let errorMessages = [];

            if (productEntries.length === 0) {
                isValid = false;
                errorMessages.push('Please add at least one product.');
            }

            productEntries.forEach((entry, index) => {
                const productSelect = entry.querySelector('.product-select');
                const quantityInput = entry.querySelector('.quantity-input');
                const priceInput = entry.querySelector('.price-input');
                const serialNumbersInput = entry.querySelector('.serial-numbers-input');
                
                const productName = productSelect.options[productSelect.selectedIndex]?.text.split(' (Stock:')[0] || `Product #${index + 1}`;
                const productId = productSelect.value;
                const quantity = parseInt(quantityInput.value) || 0;
                const unitPrice = parseFloat(priceInput.value) || 0;
                const requireSerials = productSelect.options[productSelect.selectedIndex]?.getAttribute('data-require-serials');

                if (!productId) {
                    isValid = false;
                    errorMessages.push(`For <b>${productName}</b>: Please select a product.`);
                }

                if (quantity === 0) {
                    isValid = false;
                    errorMessages.push(`For <b>${productName}</b>: Please enter a quantity.`);
                }

                if (unitPrice <= 0) {
                    isValid = false;
                    errorMessages.push(`For <b>${productName}</b>: Please enter a valid unit price.`);
                }

                if (requireSerials === '1' && quantity > 0) {
                    const serials = serialNumbersInput.value.split(/[\n,]/).map(s => s.trim()).filter(Boolean);
                    if (serials.length < quantity) {
                        isValid = false;
                        const remaining = quantity - serials.length;
                        errorMessages.push(`For <b>${productName}</b>: You have entered ${serials.length} of ${quantity} serial numbers. Please add ${remaining} more.`);
                    } else if (serials.length > quantity) {
                        isValid = false;
                        const exceeded = serials.length - quantity;
                        errorMessages.push(`For <b>${productName}</b>: You have exceeded serial numbers by ${exceeded}. Kindly check to complete.`);
                    }
                }
            });

            if (!isValid || errorMessages.length > 0) {
                e.preventDefault();
                const errorHtml = `
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <strong>Please fix the following errors:</strong><br>
                        ${errorMessages.join('<br>')}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>`;
                alertPlaceholder.innerHTML = errorHtml;
            }
        });
    }

    // Initialize with one product entry when modal opens
    if (manualAdjustmentModal) {
        manualAdjustmentModal.addEventListener('show.bs.modal', function() {
            // Clear existing entries
            productsContainer.innerHTML = '';
            productCounter = 0;
            
            // Clear alerts
            const alertPlaceholder = document.getElementById('stockAdjustmentAlerts');
            if (alertPlaceholder) {
                alertPlaceholder.innerHTML = '';
            }

            // Add initial product entry
            const productEntry = createProductEntry();
            productsContainer.appendChild(productEntry);
            
            // Initialize scroll indicator
            updateScrollIndicator();
        });
    }

    // Bulk Import Modal Enhancements
    const bulkImportModal = document.getElementById('bulkImportModal');
    const bulkFileInput = document.getElementById('bulk_stock_file');
    
    if (bulkImportModal && bulkFileInput) {
        // File input change handler
        bulkFileInput.addEventListener('change', function() {
            const file = this.files[0];
            if (file) {
                const fileName = file.name;
                const fileSize = (file.size / 1024 / 1024).toFixed(2); // Size in MB
                const fileType = file.type;
                
                // Create file info display
                let fileInfo = document.querySelector('.file-info');
                if (!fileInfo) {
                    fileInfo = document.createElement('div');
                    fileInfo.className = 'file-info mt-2 p-2 bg-light rounded';
                    this.parentNode.appendChild(fileInfo);
                }
                
                fileInfo.innerHTML = `
                    <small class="text-muted">
                        <i class="fas fa-file-excel me-1"></i>
                        <strong>${fileName}</strong> (${fileSize} MB)
                        <br>
                        <span class="text-success">✓ File ready for upload</span>
                    </small>
                `;
            }
        });

        // Form submission handler
        const bulkImportForm = bulkImportModal.querySelector('form');
        if (bulkImportForm) {
            bulkImportForm.addEventListener('submit', function(e) {
                const submitBtn = this.querySelector('button[type="submit"]');
                const fileInput = this.querySelector('input[type="file"]');
                
                if (!fileInput.files[0]) {
                    e.preventDefault();
                    alert('Please select a file to upload.');
                    return;
                }
                
                // Show loading state
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
                submitBtn.disabled = true;
                
                // Add loading overlay to modal
                const modalBody = this.closest('.modal-content');
                const overlay = document.createElement('div');
                overlay.className = 'position-absolute top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center';
                overlay.style.backgroundColor = 'rgba(255,255,255,0.8)';
                overlay.style.zIndex = '9999';
                overlay.innerHTML = `
                    <div class="text-center">
                        <div class="spinner-border text-primary mb-2" role="status"></div>
                        <div><strong>Processing bulk import...</strong></div>
                        <small class="text-muted">This may take a few moments</small>
                    </div>
                `;
                modalBody.style.position = 'relative';
                modalBody.appendChild(overlay);
            });
        }
    }

    // Live search functionality
    const searchInput = document.querySelector('input[name="search"]');
    const categorySelect = document.querySelector('select[name="category_id"]');
    const sortSelect = document.querySelector('select[name="sort"]');
    const orderSelect = document.querySelector('select[name="order"]');
    const productsTable = document.querySelector('table tbody');
    const productsCount = document.querySelector('.badge.bg-primary');

    let searchTimeout;

    function performLiveSearch() {
        const searchTerm = searchInput.value.toLowerCase();
        const categoryId = categorySelect.value;
        const allRows = productsTable.querySelectorAll('tr');
        let visibleCount = 0;

                 allRows.forEach(row => {
             const productName = row.cells[0].textContent.toLowerCase();
             const categoryName = row.cells[2].textContent.toLowerCase();
             const productDescription = row.getAttribute('data-description')?.toLowerCase() || '';
             const serialNumbers = row.getAttribute('data-serial-numbers')?.toLowerCase() || '';
             
             // Get category ID from row data attribute if available
             const rowCategoryId = row.getAttribute('data-category-id') || '';

             let matchesSearch = true;
             let matchesCategory = true;

             // Check search match
             if (searchTerm) {
                 matchesSearch = productName.includes(searchTerm) || 
                                categoryName.includes(searchTerm) ||
                                productDescription.includes(searchTerm) ||
                                serialNumbers.includes(searchTerm);
             }

            // Check category match
            if (categoryId) {
                matchesCategory = rowCategoryId === categoryId;
            }

            if (matchesSearch && matchesCategory) {
                row.style.display = '';
                visibleCount++;
            } else {
                row.style.display = 'none';
            }
        });

        // Update count
        if (productsCount) {
            productsCount.textContent = `${visibleCount} Products`;
        }
    }

    // Live search on input
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(performLiveSearch, 300); // Debounce 300ms
        });
    }

    // Category filter change
    if (categorySelect) {
        categorySelect.addEventListener('change', performLiveSearch);
    }

    // Apply Filters button functionality
    const filterForm = document.querySelector('form[method="GET"]');
    if (filterForm) {
        filterForm.addEventListener('submit', function(e) {
            // Allow normal form submission for non-search filters
            return true;
        });
    }

    // Sort functionality
    function applySorting() {
        const sortBy = sortSelect.value;
        const order = orderSelect.value;
        const tbody = productsTable;
        const rows = Array.from(tbody.querySelectorAll('tr')).filter(row => row.style.display !== 'none');

        rows.sort((a, b) => {
            let aValue, bValue;
            
            switch(sortBy) {
                case 'name':
                    aValue = a.cells[0].textContent.trim();
                    bValue = b.cells[0].textContent.trim();
                    break;
                case 'price':
                    aValue = parseFloat(a.cells[3].textContent.replace(/[^\d.]/g, '')) || 0;
                    bValue = parseFloat(b.cells[3].textContent.replace(/[^\d.]/g, '')) || 0;
                    break;
                case 'cost_price':
                    aValue = parseFloat(a.cells[4].textContent.replace(/[^\d.]/g, '')) || 0;
                    bValue = parseFloat(b.cells[4].textContent.replace(/[^\d.]/g, '')) || 0;
                    break;
                case 'quantity':
                    aValue = parseInt(a.cells[5].textContent.trim()) || 0;
                    bValue = parseInt(b.cells[5].textContent.trim()) || 0;
                    break;
                case 'times_sold':
                    aValue = parseInt(a.cells[7].textContent.trim()) || 0;
                    bValue = parseInt(b.cells[7].textContent.trim()) || 0;
                    break;
                default:
                    aValue = a.cells[0].textContent.trim();
                    bValue = b.cells[0].textContent.trim();
            }

            if (typeof aValue === 'string') {
                return order === 'desc' ? bValue.localeCompare(aValue) : aValue.localeCompare(bValue);
            } else {
                return order === 'desc' ? bValue - aValue : aValue - bValue;
            }
        });

        // Clear tbody and append sorted rows
        tbody.innerHTML = '';
        rows.forEach(row => tbody.appendChild(row));
        
        // Re-append hidden rows
        const hiddenRows = Array.from(productsTable.parentNode.querySelectorAll('tr[style*="display: none"]'));
        hiddenRows.forEach(row => tbody.appendChild(row));
    }

    // Sort and order change handlers
    if (sortSelect) {
        sortSelect.addEventListener('change', applySorting);
    }
    if (orderSelect) {
        orderSelect.addEventListener('change', applySorting);
    }


});
</script>

<?php 
// Add permission modal components
echo getUniversalPermissionModalHTML();
echo getUniversalPermissionModalJS();
?>

<?php include '../includes/footer.php'; ?>
</body>
</html> 
